/**
 * \file sdc_daemon_interface.c
 *
 * \brief Connection to SDC daemon (khd) - only required in case daemon is used
 *
 * This file contains all functions to send requests and receive response from the SDC daemon (khd).
 * SOURCES in make configuration mk/?.mk must include the sdc_daemon_interface in case the SDC
 * daemon shall be used in the corresponding architecture.
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>

#include <sdc_keystore_keys.h>
#include <private/sdc_advanced.h>
#include <sdc/daemon/arch/sdc_daemon.h>
#include <sdc/daemon/arch/sdc_daemon_arch.h>

/* Definitions types and defaults */

typedef enum {
    INSERT_KEY,
    GENERATE_RANDOM_KEY
} sdc_daemon_install_key_common_t;


/* Functions */

static sdc_error_t sdc_daemon_send_receive(
    const void *msg,     /* msg starts with sdc_daemon_msg_header_type, which includes length */
    sdc_daemon_msg_response_type *msg_response)
{
    sdc_error_t err;
    int ss_fd;
    struct sockaddr_un addr = {0};
    char *path;
    int err_chk;
    ssize_t msg_len = 0;
    sdc_daemon_msg_header_type *msg_header = (sdc_daemon_msg_header_type *)msg;


    /* Create the socket */
    ss_fd = socket(PF_LOCAL, SOCK_STREAM, 0);
    if (ss_fd == -1) {
        return SDC_DAEMON_COMMUNICATION_ERROR;
    }

    /* lookup for the UN socket */
    err = sdc_config_file_lookup(&path, SDC_CONFIG_DAEMON_SOCKET);
    if(err != SDC_OK) {
        return err;
    }

    /* Store the server's path in the socket address */
    addr.sun_family = AF_LOCAL;
    strncpy(addr.sun_path, path, sizeof(addr.sun_path)-1);

    /* Connect to socket */
    err_chk = connect(ss_fd, (struct sockaddr *)&addr, strlen(addr.sun_path) + sizeof(addr.sun_family));
    if (err_chk == -1) {
        err = SDC_DAEMON_COMMUNICATION_ERROR;
    }

    if (err == SDC_OK) {
        /* Send the message to the daemon */
        msg_len = send(ss_fd, msg, msg_header->msg_len, MSG_NOSIGNAL);
        if (msg_len == -1) {
            err = SDC_DAEMON_COMMUNICATION_ERROR;
        } else if ((size_t)msg_len != msg_header->msg_len) {
            err = SDC_DAEMON_COMMUNICATION_ERROR;
        }
    }

    if (err == SDC_OK) {
        /* Read the response from the daemon */
        msg_len = recv(ss_fd, msg_response, sizeof(sdc_daemon_msg_response_type), 0);

        /* Check the response */
        if (msg_len <= 0) {

            err = SDC_DAEMON_INVALID_RESPONSE;

        } else if ((size_t)msg_len != sizeof(sdc_daemon_msg_response_type)) {

            err = SDC_DAEMON_INVALID_RESPONSE;

        } else if (msg_response->msg_len != sizeof(sdc_daemon_msg_response_type)) {

            err = SDC_DAEMON_INVALID_RESPONSE;

        } else if (msg_response->type == MSG_ERROR) {

            err = msg_response->value.sdc_error;

        } else if (msg_response->type != MSG_SUCCESS) {

            /* unexpected response */
            err = SDC_DAEMON_INVALID_RESPONSE;

        }
    }

    /* Close the socket */
    close(ss_fd);

    free(path);

    return err;
}

static sdc_error_t sdc_daemon_install_key_common(
    const sdc_daemon_install_key_common_t install_type,
    const sdc_keystore_key_cfg_t *key_config,
    const sdc_keysecret_t *key_secret)
{
    sdc_daemon_msg_header_type *msg_header;
    sdc_daemon_msg_install_type *msg_install;
    sdc_daemon_msg_response_type msg_response;
    sdc_error_t err;
    size_t msg_buf_len;
    void *msg_key;
    uint8_t *msg_buf;


    /* for keystore keys uid is limited to current euid */
    if (key_config->common.perms->uid != geteuid()) {
        return SDC_PERM_INVALID;
    }

    /* Calculate buffer length for message */
    msg_buf_len = sizeof(sdc_daemon_msg_header_type) +
                  sizeof(sdc_daemon_msg_install_type);

    /* calloc and NULL check had to be doubled to get rid of QAC warning */
    switch(install_type) {
    case INSERT_KEY:
        /* check if key_length will fit into buffer */
        if (!key_secret)
            return SDC_IN_DATA_INVALID;

        if (key_secret->secret_len > (SDC_DAEMON_MAX_MSG_LEN - msg_buf_len)) {
            return SDC_KEYLEN_INVALID;
        }

        /* add length of key data */
        msg_buf_len += key_secret->secret_len;

        /* Allocate and zero buffer */
        msg_buf = calloc(1, msg_buf_len);

        if (msg_buf == NULL) {
            return SDC_NO_MEM;
        }
        msg_key = (void *) (msg_buf + sizeof(sdc_daemon_msg_header_type) + sizeof(sdc_daemon_msg_install_type));

        break;
    case GENERATE_RANDOM_KEY:
        /* Allocate and zero buffer */
        msg_buf = calloc(1, msg_buf_len);

        if (msg_buf == NULL) {
            return SDC_NO_MEM;
        }

        msg_key = NULL;
        break;
    default:
        return SDC_UNKNOWN_ERROR;
    }

    /* Index message components into message buffer */
    msg_header  = (void *) msg_buf;
    msg_install = (void *) (msg_buf + sizeof(sdc_daemon_msg_header_type));

    /* Build message header */
    msg_header->msg_len = msg_buf_len;
    msg_header->operation = MSG_INSTALL_KEY;

    /* Build install-specific message part */
    if (key_config->key_stor_opt == SDC_VOLATILE_STORAGE_KEY)
        msg_install->flags |= MSG_FLAG_VOLATILE;
    if (key_config->kid_opt == SDC_CREATE_KEY_FIXED_ID) {
        msg_install->req_id_min = *(key_config->kid);
        msg_install->req_id_max = msg_install->req_id_min;
    } else {
        msg_install->req_id_min = SDC_AUTOMATIC_KID_RANGE_MIN;
        msg_install->req_id_max = SDC_AUTOMATIC_KID_RANGE_MAX;
    }
    msg_install->fmt = key_config->common.fmt;
    msg_install->len = key_config->common.len;
    msg_install->key_data_len = (key_secret ? key_secret->secret_len : 0);
    msg_install->key_data_enc = (key_secret ? key_secret->enc :
                                 SDC_KEY_ENC_UNKNOWN);

    /* install_type == INSERT_UNENCRYPTED_KEY <=> msg_key != NULL */
    if (msg_key != NULL && key_secret) {
        /* Add plain key data as final message component */
        /* Note : this is an internal function so the plain key can't be NULL */
        memcpy(msg_key, key_secret->secret, key_secret->secret_len);
    } else {
        /* install_type == GENERATE_RANDOM_KEY is the only other option */
        /* set flag */
        msg_install->flags |= MSG_FLAG_RANDOM_KEY;
    }

    memcpy(&msg_install->permissions,
           key_config->common.perms,
           sizeof(sdc_permissions_t));

    err = sdc_daemon_send_receive(msg_buf, &msg_response);

    if (err == SDC_OK) {
        /* msg_response.type == MSG_SUCCESS */
        /* Key installed, pass ID back to caller */
        *(key_config->kid) = msg_response.value.installed_id;
    }

    free(msg_buf);

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_insert_storage_key(
    const sdc_keystore_key_cfg_t *key_config,
    const sdc_keysecret_t *key_secret)
{
    return sdc_daemon_install_key_common(
               INSERT_KEY,
               key_config,
               key_secret);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_generate_sym_storage_key(
    const sdc_keystore_key_cfg_t *key_config)
{
    return sdc_daemon_install_key_common(
               GENERATE_RANDOM_KEY,
               key_config,
               NULL);
}

static sdc_error_t sdc_daemon_import_key(
    const sdc_keystore_key_cfg_t *key_config,
    const sdc_keysecret_t *key_secret_packed)
{
    sdc_daemon_msg_header_type *msg_header;
    sdc_daemon_msg_import_type *msg_import;
    sdc_daemon_msg_response_type msg_response;
    sdc_error_t err;
    size_t msg_buf_len;
    void *msg_key;
    uint8_t *msg_buf;

    /* for keystore keys uid is limited to current euid */
    if (key_config->common.perms->uid != geteuid()) {
        return SDC_PERM_INVALID;
    }

    /* Calculate buffer length for message */
    msg_buf_len = sizeof(sdc_daemon_msg_header_type) +
                  sizeof(sdc_daemon_msg_install_type);

    /* check if key_length will fit into buffer */
    if (!key_secret_packed)
        return SDC_IN_DATA_INVALID;

    if (key_secret_packed->secret_len > (SDC_DAEMON_MAX_MSG_LEN - msg_buf_len)) {
        return SDC_KEYLEN_INVALID;
    }

    /* add length of key data */
    msg_buf_len += key_secret_packed->secret_len;

    /* Allocate and zero buffer */
    msg_buf = calloc(1, msg_buf_len);

    if (msg_buf == NULL) {
        return SDC_NO_MEM;
    }
    msg_key = (void *) (msg_buf + sizeof(sdc_daemon_msg_header_type) + sizeof(sdc_daemon_msg_import_type));

    /* Index message components into message buffer */
    msg_header  = (void *) msg_buf;
    msg_import = (void *) (msg_buf + sizeof(sdc_daemon_msg_header_type));

    /* Build message header */
    msg_header->msg_len = msg_buf_len;
    msg_header->operation = MSG_IMPORT_KEY;

    /* Build install-specific message part */
    if (key_config->key_stor_opt == SDC_VOLATILE_STORAGE_KEY)
        msg_import->flags |= MSG_FLAG_VOLATILE;
    if (key_config->kid_opt == SDC_CREATE_KEY_FIXED_ID) {
        msg_import->req_id_min = *(key_config->kid);
        msg_import->req_id_max = msg_import->req_id_min;
    } else {
        msg_import->req_id_min = SDC_AUTOMATIC_KID_RANGE_MIN;
        msg_import->req_id_max = SDC_AUTOMATIC_KID_RANGE_MAX;
    }
    msg_import->fmt = key_config->common.fmt;
    msg_import->len = key_config->common.len;
    msg_import->key_data_len = key_secret_packed->secret_len;
    msg_import->key_data_enc = key_secret_packed->enc;

    /* Add key data as final message component */
    /* Note : this is an internal function so the plain key can't be NULL */
    memcpy(msg_key, key_secret_packed->secret, key_secret_packed->secret_len);

    memcpy(&msg_import->permissions,
           key_config->common.perms,
           sizeof(sdc_permissions_t));

    err = sdc_daemon_send_receive(msg_buf, &msg_response);

    if (err == SDC_OK) {
        /* msg_response.type == MSG_SUCCESS */
        /* Key installed, pass ID back to caller */
        *(key_config->kid) = msg_response.value.installed_id;
    }

    free(msg_buf);

    return err;
}

sdc_error_t sdc_arch_import_wrapped_storage_key(
    sdc_session_t *session,
    const sdc_keystore_key_cfg_t *key_config,
    const sdc_wrapped_key_t *wrapped_key)
{
    sdc_error_t err = SDC_OK;
    uint8_t *daemon_data = NULL;
    size_t daemon_data_len = 0;

    err = sdc_arch_wrapped_storage_key_to_daemon_data(
        session, key_config,
        wrapped_key,
        &daemon_data, &daemon_data_len);

    if ((err == SDC_OK) && (daemon_data != NULL) && (daemon_data_len > 0)) {
        sdc_keysecret_t key_secret_packed = {
                .enc = wrapped_key->enc,
                .secret = daemon_data,
                .secret_len = daemon_data_len,
        };

        err = sdc_daemon_import_key(key_config, &key_secret_packed);
    }

    free(daemon_data);

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_remove_storage_key(sdc_key_id_t kid)
{
    sdc_daemon_msg_header_type *msg_header;
    sdc_daemon_msg_remove_type *msg_remove;
    sdc_daemon_msg_response_type msg_response;
    sdc_error_t err;
    size_t msg_buf_len;
    uint8_t *msg_buf;

    /* Calculate buffer length for message */
    msg_buf_len = sizeof(sdc_daemon_msg_header_type) + sizeof(sdc_daemon_msg_remove_type);

    /* Allocate and zero buffer */
    msg_buf = calloc(1, msg_buf_len);
    if (msg_buf == NULL) {
        return SDC_NO_MEM;
    }

    /* Index message components into message buffer */
    msg_header = (sdc_daemon_msg_header_type *) msg_buf;
    msg_remove = (sdc_daemon_msg_remove_type *) (msg_buf + sizeof(sdc_daemon_msg_header_type));

    /* Build message header */
    msg_header->msg_len = sizeof(sdc_daemon_msg_header_type) + sizeof(sdc_daemon_msg_remove_type);
    msg_header->operation = MSG_REMOVE_KEY;

    /* Build remove-specific message part */
    msg_remove->id = kid;

    err = sdc_daemon_send_receive(msg_buf, &msg_response);
    /* no content in response */

    free(msg_buf);

    return err;
}

sdc_error_t sdc_daemon_check_permission_range(const sdc_permissions_t *perms)
{
    return sdc_intern_check_permission_range(perms);
}
